-- DROP SCHEMA IF EXISTS tcache CASCADE;


--
CREATE TABLE tcache.function_cache (
  c_funcname    VARCHAR(100),
  c_param0      VARCHAR(100),
  c_param1      VARCHAR(100),
  c_param2      VARCHAR(100),
  c_param3      VARCHAR(100),
  c_param4      VARCHAR(100),
  c_param5      VARCHAR(100),
  c_param6      VARCHAR(100),
  c_dirty       BOOL,
  c_resultn     NUMERIC(16,4),
  c_resultc     VARCHAR
);

-- nicht nehmen, da nicht für UPDATE und DELETE Fall optimiert, wenn keine NOT c_dirty Bedingung existiert.
-- CREATE INDEX function_cache_index_valid ON tcache.function_cache(c_funcname, c_param0) WHERE NOT c_dirty;

-- Indizes
    CREATE INDEX function_cache_index ON tcache.function_cache(c_funcname, c_param0) INCLUDE ( c_resultn, c_resultc);
--


-- Cache schreiben
  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache(
        funcname VARCHAR(100),
        param0 VARCHAR(100),    param1 VARCHAR(100),  param2 VARCHAR(100),  param3 VARCHAR(100),
        param4 VARCHAR(100),    param5 VARCHAR(100),  param6 VARCHAR(100),
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      UPDATE tcache.function_cache SET
        c_resultn = resultn,
        c_resultc = resultc,
        c_dirty   = false
      WHERE funcname            = c_funcname
        AND param0              = c_param0
        AND COALESCE(param1,'') = COALESCE(c_param1,'')
        AND COALESCE(param2,'') = COALESCE(c_param2,'')
        AND COALESCE(param3,'') = COALESCE(c_param3,'')
        AND COALESCE(param4,'') = COALESCE(c_param4,'')
        AND COALESCE(param5,'') = COALESCE(c_param5,'')
        AND COALESCE(param6,'') = COALESCE(c_param6,'')
      ;

      IF NOT found THEN
          INSERT INTO tcache.function_cache
          SELECT funcname, param0, param1, param2, param3, param4, param5, param6, false, resultn, resultc;
      END IF;

      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_1param(
        funcname VARCHAR(100),
        param0 VARCHAR(100),
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setcache(funcname, param0, NULL, NULL, NULL, NULL, NULL, NULL, resultn, resultc);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_1param(
        funcname VARCHAR(100),
        param0 INTEGER,
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
       PERFORM tcache.function_cache_setcache(funcname, param0::VARCHAR, NULL, NULL, NULL, NULL, NULL, NULL, resultn, resultc);
       RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_2param(
        funcname VARCHAR(100),
        param0 VARCHAR(100),    param1 VARCHAR(100),
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setcache(funcname, param0, param1, NULL, NULL, NULL, NULL, NULL, resultn, resultc);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_2param(
        funcname VARCHAR(100),
        param0 INTEGER,         param1 VARCHAR(100),
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setcache(funcname, param0::VARCHAR, param1, NULL, NULL, NULL, NULL, NULL, resultn, resultc);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_2param(
        funcname VARCHAR(100),
        param0 INTEGER,         param1 INTEGER,
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setcache(funcname, param0::VARCHAR, param1::VARCHAR, NULL, NULL, NULL, NULL, NULL, resultn, resultc);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  -- #8012
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_3param(
        funcname VARCHAR(100),
        param0 VARCHAR(100),    param1 VARCHAR(100),  param2 VARCHAR(100),
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setcache(funcname, param0, param1, param2, NULL, NULL, NULL, NULL, resultn, resultc);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setcache_3param(
        funcname VARCHAR(100),
        param0 INTEGER,         param1 VARCHAR(100),  param2 VARCHAR(100),
        resultn NUMERIC(16,4),  resultc VARCHAR(200) DEFAULT NULL
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setcache(funcname, param0::VARCHAR, param1, param2, NULL, NULL, NULL, NULL, resultn, resultc);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --
--


--Cache ungültig setzen
  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setdirty(
        funcname VARCHAR(100),
        param0 VARCHAR(100),    param1 VARCHAR(100),  param2 VARCHAR(100),  param3 VARCHAR(100),
        param4 VARCHAR(100),    param5 VARCHAR(100),  param6 VARCHAR(100)
    ) RETURNS VOID AS $$
    BEGIN
      UPDATE tcache.function_cache SET
        c_dirty = true
      WHERE funcname            = c_funcname
        AND param0              = c_param0
        AND COALESCE(param1,'') = COALESCE(c_param1,'')
        AND COALESCE(param2,'') = COALESCE(c_param2,'')
        AND COALESCE(param3,'') = COALESCE(c_param3,'')
        AND COALESCE(param4,'') = COALESCE(c_param4,'')
        AND COALESCE(param5,'') = COALESCE(c_param5,'')
        AND COALESCE(param6,'') = COALESCE(c_param6,'')
        AND NOT c_dirty
      ;

      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setdirty_1param(
        funcname VARCHAR(100),
        param0  VARCHAR(100)
    ) RETURNS VOID AS $$
    BEGIN
      UPDATE tcache.function_cache SET
        c_dirty = true
      WHERE funcname = c_funcname
        AND param0   = c_param0
        AND NOT c_dirty
      ;

      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setdirty_1param(
        funcname VARCHAR(100),
        param0  INTEGER
    ) RETURNS VOID AS $$
    BEGIN
      UPDATE tcache.function_cache SET
        c_dirty = true
      WHERE funcname = c_funcname
        AND param0   = c_param0
        AND NOT c_dirty
      ;

      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setdirty_2param(
        funcname VARCHAR(100),
        param0 VARCHAR(100),
        param1 VARCHAR(100)
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setdirty(funcname, param0, param1, null, null, null, null, null);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setdirty_2param(
        funcname VARCHAR(100),
        param0 INTEGER,
        param1 VARCHAR(100)
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setdirty(funcname, param0::VARCHAR, param1, null, null, null, null, null);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  --
  CREATE OR REPLACE FUNCTION tcache.function_cache_setdirty_2param(
        funcname VARCHAR(100),
        param0 INTEGER,
        param1 INTEGER
    ) RETURNS VOID AS $$
    BEGIN
      PERFORM tcache.function_cache_setdirty(funcname, param0::VARCHAR, param1::VARCHAR, null, null, null, null, null);
      RETURN;
    END $$ LANGUAGE plpgsql;
  --
--

  CREATE OR REPLACE FUNCTION TSystem.string_to_int_hash(text) RETURNS bigint AS $$
      --https://stackoverflow.com/questions/9809381/hashing-a-string-to-a-numeric-value-in-postgresql
      SELECT ('x' || substr( md5($1), 1, 16) )::bit(64)::bigint;
      $$ LANGUAGE sql IMMUTABLE;


-- Cache-Views für TCachedSQL - siehe Kommentare an TCachedSQL (UProdatClasses)
-- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/DB-Cache_(Assistenten)
-- http://redmine.prodat-sql.de/issues/8348
  -- Aufruf beim Start des AppServers
  CREATE OR REPLACE FUNCTION TCache.CachedViews_Clear() RETURNS BOOLEAN AS $$
    DECLARE
        r RECORD;
        e BOOLEAN;
    BEGIN
      e := true;

      PERFORM TSystem.LogDebug( 'BEGIN', logtype => 'pltProfiling' );

      FOR r IN
          SELECT nspname, relname
            FROM pg_class
            LEFT JOIN pg_namespace ON pg_namespace.oid = relnamespace
           WHERE relkind = 'm'
             AND nspname ILIKE 'TCache'
           ORDER BY relname
      LOOP
          BEGIN
              PERFORM TSystem.LogInfo( 'DROP ' || r.nspname || '.' || r.relname );
              -- EXECUTE 'DROP MATERIALIZED VIEW IF EXISTS ' || r.nspname || '.' || r.relname || ' CASCADE';
              EXECUTE 'DROP MATERIALIZED VIEW ' || r.nspname || '.' || r.relname;
          EXCEPTION WHEN others THEN
              PERFORM TSystem.LogFatalError(
                          TSystem.LogFormat(            -- Message
                              'ERROR ON DROP ' || r.nspname || '.' || r.relname
                            , false                     -- Zeilenumbruch
                            , 0
                            , SQLERRM
                          )
                      )
              ;
              e := false;
          END;
      END LOOP;

      PERFORM TSystem.LogDebug( 'END', logtype => 'pltProfiling' );

      RETURN e;
    END $$ LANGUAGE plpgsql;
  --

  -- Aufruf im RefreshZyklus des AppServers > TProdatSrvXEService.BackgroundThread
   -- ACHTUNG: Logik wie FUNCTION tartikel.bestand_abgleich__all__background()
    CREATE OR REPLACE FUNCTION TCache.CachedViews_Refresh() RETURNS BOOLEAN AS $$
    DECLARE
        r                         record;
        function_name_int_hash    bigint;
        e                         boolean;
        ident                     varchar;
        I                         integer;
        APPNAME                   varchar = 'PRODAT-ERP-CACHE(' || current_database() || ')';
        ConnectionName_BaseName   varchar = 'CachedViews_Refresh_';
        ConnectionName_Free       varchar;
        num_connections           integer = 4;
        debug_s                   varchar;
    BEGIN
      e := true;

      function_name_int_hash := TSystem.string_to_int_hash('CachedViews_Refresh');

      -- https://redmine.prodat-sql.de/issues/15775
      IF NOT pg_try_advisory_xact_lock(function_name_int_hash) THEN
         PERFORM TSystem.LogWarning( 'pg_try_advisory_xact_lock failed:' || ServerViewName_int_hash);
         RETURN null;
      END IF;

      PERFORM TSystem.LogAlways( 'BEGIN', logtype =>  'pltProfiling' );

      PERFORM pg_terminate_backend(pid)
         FROM pg_stat_activity
        WHERE datname = current_database()
          AND application_name = APPNAME;

      --Connections aufbauen
      FOR i IN 0..num_connections-1 LOOP
          ConnectionName_Free :=  ConnectionName_BaseName || current_database() || '_' || chr(65 + I); -- Buchstaben A..D

          -- Falls Connection aus irgendeinem Grund noch vorhanden, jetzt hart schliessen
          -- TODO: einschränken auf aktuellen Connectinoname, damit es nicht zB bestand_abgleich__all__background Connections abschiesst
          IF ConnectionName_Free = ANY(dblink_get_connections()) THEN
              RAISE WARNING 'TCache.CachedViews_Refresh: LOOP=%, Connection still active, closing:"%"    (Connections:%)', i, ConnectionName_Free, dblink_get_connections();
              BEGIN
                  PERFORM dblink_disconnect( ConnectionName_Free );
              EXCEPTION WHEN others THEN
                  RAISE WARNING 'cachedviews_refresh(): ERROR ON dblink_disconnect %', ConnectionName_Free USING DETAIL = SQLERRM;
              END;
          END IF;

          -- Connection aufbauen und Statement Timeout
          PERFORM dblink_connect_u(ConnectionName_Free, 'dbname=' || current_database() || /*' host='||inet_client_addr()||*/' port=' || inet_server_port() || ' user=APPS password=Application-Server application_name=' || quote_literal(APPNAME));
          PERFORM dblink_exec(ConnectionName_Free, 'SET statement_timeout=30000', true);
      END LOOP;

      -- durch alle View laufen und Refresh
      FOR r IN
          SELECT nspname, relname -- Abfrage Views
            FROM pg_class
            LEFT JOIN pg_namespace ON pg_namespace.oid = relnamespace
           WHERE relkind = 'm'
             AND nspname ILIKE 'TCache'
           ORDER BY relname
      LOOP
          BEGIN
              -- Name Cached View, welche aktualisiert werden soll
              ident := r.nspname || '.' || r.relname;

              -- Endlosschleife, wird durch EXIT verlassen. Connection abwarten und Asyncrones View Refresh aufrufen
              LOOP
                  FOR i IN 0..num_connections-1 LOOP -- versuchen eine freie Connection zu bekommen
                      ConnectionName_Free := 'CachedViews_Refresh_' || current_database() || '_' || chr(65 + I);

                      IF dblink_is_busy(ConnectionName_Free) = 0 THEN
                          PERFORM * FROM dblink_get_result(ConnectionName_Free, false) AS CachedViews_Refresh(Result VARCHAR); -- GetResult

                          -- Once again >> This function must be called if dblink_send_query returned 1.
                          -- It must be called once for each query sent, and one additional time to obtain an empty set result, before the connection can be used again.
                          PERFORM * FROM dblink_get_result(ConnectionName_Free, false) AS CachedViews_Refresh(Result VARCHAR);

                          -- RAISE NOTICE 'TRY EXECUTE ON % - %', ConnectionName_Free, ident;

                          PERFORM dblink_send_query( ConnectionName_Free, 'SELECT TCache.CachedViews_Refresh(' || quote_literal(ident) || ', true)' ) AS CachedViews_Refresh;

                          -- RAISE NOTICE 'EXECUTE ON %', ConnectionName_Free;

                          EXIT; -- Connection bekommen, Schleife verlassen
                      END IF;

                      ConnectionName_Free := 'X';
                  END LOOP;


                  -- Wenn keine Verbindung erhalten,
                  -- dann auf Ausführung der laufenden Refreshs warten (gemittelt) und erneut Connection versuchen.
                  IF ConnectionName_Free = 'X' THEN

                      -- RAISE NOTICE 'TCache.CachedViews_Refresh: No Free Connection';

                      PERFORM pg_sleep( 0.1 );

                  -- Sonst Verbindung für async. Refresh des VIEWs erhalten und nächsten VIEW angehen.
                  ELSE
                      EXIT;
                  END IF;

              END LOOP; -- Endlosschleife, muss durch EXIT verlassen werden

          EXCEPTION WHEN others THEN
              -- RAISE WARNING 'cachedviews_refresh(): ERROR ON REFRESH %', ident USING DETAIL = SQLERRM;
              PERFORM TSystem.LogError(
                          TSystem.LogFormat(              -- Message
                              'ERROR ON CachedViews_Refresh ' || ident
                             , false          -- Zeilenumbruch
                             , 0
                             , SQLERRM
                          )
                      )
              ;
              e := false;
          END;
      END LOOP;

      -- alle Connections schließen
      FOR i IN 0..num_connections-1 LOOP
          ConnectionName_Free := 'CachedViews_Refresh_' || current_database() || '_' || chr(65 + I);

          -- Ergebnis abrufen. False => Ignore Error
          PERFORM * FROM dblink_get_result(ConnectionName_Free, false) AS CachedViews_Refresh(Result VARCHAR); -- GetResult

          -- Once again >> This function must be called if dblink_send_query returned 1.
          -- It must be called once for each query sent, and one additional time to obtain an empty set result, before the connection can be used again.
          PERFORM * FROM dblink_get_result(ConnectionName_Free, false) AS CachedViews_Refresh(Result VARCHAR);

          PERFORM dblink_disconnect(ConnectionName_Free);
      END LOOP;

      PERFORM TSystem.LogAlways( 'END', logtype => 'pltProfiling' );

      RETURN e;
    END $$ LANGUAGE plpgsql;
  --

  -- Erzeugen des Views
  -- TCachedSQL anschauen: cCreateStatement. ******Noch nicht in Verwendung bisher!!!***** Klärung Schema automatisch wie unten bei "CachedViews_Refresh"? -> ServerViewNameSpace
  CREATE OR REPLACE FUNCTION TCache.CachedViews_Create(ServerViewName varchar, ServerViewNameSpace varchar, sql_statement text) RETURNS void
    AS $$
    DECLARE
        ServerViewName_int_hash    bigint;
    BEGIN
      -- falls dieser View bereits durch einen anderen Client in Aktualisierung ist
      ServerViewName_int_hash := TSystem.string_to_int_hash(ServerViewName);
      IF NOT pg_try_advisory_xact_lock(ServerViewName_int_hash) THEN
         PERFORM TSystem.LogWarning( 'pg_try_advisory_xact_lock failed:' || ServerViewName_int_hash);
         RETURN;
      END IF;

      ServerViewName := concat_ws('.', ServerViewNameSpace, ServerViewName);

      PERFORM TSystem.LogAlways( 'BEGIN:' || ServerViewName, logtype => 'pltProfiling' );

      BEGIN
          EXECUTE 'CREATE MATERIALIZED VIEW ' || ServerViewName || ' AS ' || sql_statement;
          EXECUTE ' ALTER MATERIALIZED VIEW ' || ServerViewName || ' OWNER TO "SYS.Prodat-User"';
      EXCEPTION WHEN others THEN
          PERFORM TSystem.LogError(
                      TSystem.LogFormat(              -- Message
                          'ERROR ON CREATE ' || ServerViewName
                         , false          -- Zeilenumbruch
                         , 0
                         , SQLERRM
                      )
                  )
          ;
      END;

      PERFORM TSystem.LogAlways( 'END:' || ServerViewName, logtype => 'pltProfiling' );

    END $$ LANGUAGE plpgsql;
  --

  -- Manueller Refresh, z.B. aus irgendwelchen Modulen/Triggern
  -- Beachte TCachedSQL - womit der Client die Views auch einzeln Aktualisieren kann > siehe cRefreshStatement
  CREATE OR REPLACE FUNCTION TCache.CachedViews_Refresh(ServerViewName varchar, _concurrently boolean = false) RETURNS void AS $$
    DECLARE
        ServerViewName_int_hash    bigint;
    BEGIN
      -- falls dieser View bereits durch einen anderen Client in Aktualisierung ist
      -- https://redmine.prodat-sql.de/issues/15775
      ServerViewName_int_hash := TSystem.string_to_int_hash(ServerViewName);
      IF NOT pg_try_advisory_xact_lock(ServerViewName_int_hash) THEN
         PERFORM TSystem.LogWarning( 'pg_try_advisory_xact_lock failed:' || ServerViewName_int_hash);
         RETURN;
      END IF;

      PERFORM TSystem.LogAlways( 'BEGIN:' || ServerViewName, logtype => 'pltProfiling' );

      IF ServerViewName NOT ILIKE 'TCache.%' THEN
          ServerViewName := 'TCache.' || ServerViewName;
      END IF;

      _concurrently := false; -- requires unique index

      BEGIN
          EXECUTE 'REFRESH MATERIALIZED VIEW ' || IFTHEN(_concurrently, ' CONCURRENTLY ', '') || ServerViewName;
      EXCEPTION WHEN others THEN
          PERFORM TSystem.LogError(
                      TSystem.LogFormat(              -- Message
                          'ERROR ON REFRESH ' || ServerViewName
                        , false     -- Zeilenumbruch
                        , 0
                        , SQLERRM
                      )
                  )
          ;
      END;

      PERFORM TSystem.LogAlways( 'END:' || ServerViewName, logtype => 'pltProfiling' );

    END $$ LANGUAGE plpgsql;
  --